#include "random_coord.h"
#include <math.h>
#include <stdlib.h>

#define G 6.67300E-11
#define PI 3.14159265
#define SQ(X) ((X)*(X))

/*
 * Three functions to generate spherical coordinates according to the Hernquist model
 */

double compute_random_radius(gal_system* sys) {
	double uniform_rand_max;	// The superior boundary of the following random variable
	double uniform_rand_var;	// The random variable associated to the uniform law in [0,uniform_rand_max]

	double radius_max = sys->radius_max;
	double typical_length = sys->typical_length;
	
	uniform_rand_max = SQ(radius_max) / SQ(radius_max + typical_length);
	uniform_rand_var = (rand() / (double) RAND_MAX) * uniform_rand_max;

	return typical_length * sqrt(uniform_rand_var) / (1 - sqrt(uniform_rand_var));
}

double compute_random_phi() {
	return (rand() / (double) RAND_MAX) * 2 * PI;
}

double compute_random_theta() {
	return (rand() / (double) RAND_MAX) * PI;
}

/*
 * Compute the probability density function of the Hernquist model
 */

double compute_hernquist_probability_density_function(double normalized_energy) {
	double q = normalized_energy;
	return ( 3*asin(q) + q*sqrt(1-SQ(q))*(1-2*SQ(q))*(8*pow(q,4)-8*SQ(q)-3) ) / pow(1-SQ(q),5/2);
}

/*
 * Compute the linear envelop density function for the following acception-rejection method
 */

double compute_linear_envelop_probability_function(double normalized_energy, double normalized_energy_max, double density_function_max) {
	return density_function_max * normalized_energy / normalized_energy_max;
}

/*
 * Generate norms of velocities, knowing the position, according to the Hernquist model
 * The acception-rejection is employed, with a linear envelop function (the Hernquist distribution is convexe)
 */

double compute_random_velocity_norm(gal_system* sys, double radius) {
	double typical_length = sys->typical_length;

	// First, we define the two random variable that we need for the acception-rejection method
	double random_normalized_energy;	// Associated with the linear envelop law 
	double random_test_value;		// Associated with the uniform law in [0,1]
	
	// Then, we define the range of the different variables
	double normalized_energy_max = sqrt( radius / (radius+typical_length) );
	double density_function_max = compute_hernquist_probability_density_function(normalized_energy_max);
	double test_value_max;

	// We can now compute our random variables
	int rejected = 1;
	while( rejected ) {
		random_normalized_energy = normalized_energy_max * sqrt( rand() / (double) RAND_MAX );
		random_test_value = ( rand() / (double) RAND_MAX );
		
		test_value_max = compute_hernquist_probability_density_function(random_normalized_energy) / compute_linear_envelop_probability_function(random_normalized_energy, normalized_energy_max, density_function_max);
		if( random_test_value <= test_value_max ) {
			rejected = 0;
		}
	}
	
	// Then, we retrieve the norm of the velocity v
	double vel_max = sqrt( 2*G*sys->total_mass / (radius+typical_length) );
	return vel_max * sqrt( 1 - SQ( random_normalized_energy / normalized_energy_max ) );
}
