#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "max.h"

#define SQ(X) ((X)*(X))

/**
 * Translate units.
 *
 * @param s a string containing a number with, possibly, a unit. The unit
 * might be s (second), m (minute), h (hour), d (day) or y (year).
 * 
 * @return a double with number in seconds.
 */
double strunit(char *s) {
        int l = strlen(s);
        double fact = 1;
        switch(s[l-1]) {
                case 'y':
                        fact *= 365.24;
                case 'd':
                        fact *= 24;
                case 'h':
                        fact *= 60;
                case 'm':
                        fact *= 60;
                case 's':
                        s[l-1] = '\0';
        }
        return fact * strtod(s, NULL);
}


void print_usage() {
	printf("Usage: perihelion [options] [filename]\n"
	"Options:\n"
	"  -o		set the output prefix (default : output)\n"
	"  -u,--unit	set the output time unit (default : 1y) (units admitted : s, m, d, y)\n"
	"  -w,--window	size of the window for the search of local maximum (default : 1d) (units admitted : s, m, d, y)\n"
	"  -h,--help	print this screen\n");
}

int main(int argc, char* argv[]) {
	int i=1;
	char* output_unit = strdup("1y");
	int fact_unit = strunit(output_unit);
	double window = 86400.0;
	char* output_prefix = strdup("output");
	char output_name[1024];
	FILE *input = NULL;
	FILE *output = NULL;
	
	while( i < argc) {
		if(!strcmp("-o",argv[i])) {
			output_prefix = strdup(argv[++i]);
			i++;
			continue;
		}
		if(!strcmp("-u",argv[i]) || !strcmp("--unit",argv[i])) {
			free(output_unit);
			output_unit = strdup(argv[++i]);
			fact_unit = strunit(argv[i]);
			if(fact_unit == 0) {
				fprintf(stderr,"error: unit null");
				return 1;
			}
			i++;
			continue;
		}
		if(!strcmp("-w",argv[i]) || !strcmp("--window",argv[i])) {
			window = strunit(argv[++i]);
			i++;
			continue;
		}
		if(!strcmp("-h",argv[i]) || !strcmp("--help",argv[i])){
			print_usage();
			return 0;
		}
		if(input != NULL) {
			print_usage();
			return 1;
		}
		input = fopen(argv[i],"r");
		if(input == NULL) {
			fprintf(stderr,"can't open: %s\n",argv[i]);
			return 1;
		}
		i++;
	}



	double step;
	if(fscanf(input,"#step=%lf\n",&step)!=1) {
		printf("Erreur: fichier incorrect");
		return 1;
	}
	sprintf(output_name,"%s_perihelion.dat",output_prefix);
	output = fopen(output_name, "w");
	if(output == NULL) {
		fprintf(stderr,"can't open %s\n",output_name);
		return 1;
	}
	fprintf(output, "#Unit: %s\n", output_unit);

	int N_local = 2*floor(window/(2*step)); // N_local has to be even
	if(N_local<3) {
		printf("Erreur: window value too small");
		fclose(input);
		return 1;
	}
	printf("N_local: %d\n",N_local);

	double pos_read[3];
	double* pos_block[3];
	for(i=0;i<3;i++)
		pos_block[i]=malloc(N_local*sizeof(double));
	double* radii = malloc(N_local*sizeof(double));
	int cur_index_max;
	int previous_index_max;
	int num_block = 0;
	int index_block;

	// We fill the first block
	index_block = 0;
	while( fscanf(input,"%lg\t%lg\t%lg\n",&pos_read[0],&pos_read[1],&pos_read[2])==3 &&
	       index_block<N_local ) {
		for(i=0;i<3;i++)
			pos_block[i][index_block] = pos_read[i];
		for(i=0;i<3;i++)
			radii[index_block] += SQ(pos_block[i][index_block]);
		radii[index_block] = sqrt(radii[index_block]);
		index_block++;
	}
	// We start the search over blocks of size N_local
	while(fscanf(input,"%lg\t%lg\t%lg\n",&pos_read[0],&pos_read[1],&pos_read[2])==3) {
		if( index_block == N_local ) {
			// We search a maximum over the block
			if(found_local_maximum(radii, N_local, &cur_index_max) &&
			   cur_index_max != previous_index_max-N_local/2) {
				fprintf(output,"%lf\t%lg\t%lg\t%lg\n"
					,step/fact_unit*(num_block*N_local/2+cur_index_max)
					,radii[cur_index_max]
					,acos(pos_block[2][cur_index_max]/radii[cur_index_max])
					,acos(pos_block[0][cur_index_max]/sqrt(SQ(pos_block[0][cur_index_max])+SQ(pos_block[1][cur_index_max]))));
				previous_index_max = cur_index_max;
			}

			// We transfer the second half in the first half
			for(i=0;i<3;i++) {
				for(index_block=0; index_block<N_local/2;index_block++) {
					pos_block[i][index_block] = pos_block[i][index_block+N_local/2];
				}
			}
			for(index_block=0;index_block<N_local/2;index_block++)
				radii[index_block] = radii[index_block+N_local/2];

			for(i=0;i<3;i++)
				pos_block[i][index_block] = pos_read[i];
			for(i=0;i<3;i++)
				radii[index_block] += SQ(pos_block[i][index_block]);
			radii[index_block] = sqrt(radii[index_block]);

			index_block++;
			num_block++;
		}
		else {
			for(i=0;i<3;i++)
				pos_block[i][index_block] = pos_read[i];
			for(i=0;i<3;i++)
				radii[index_block] += SQ(pos_block[i][index_block]);
			radii[index_block] = sqrt(radii[index_block]);
			index_block++;
		}
	}
	fclose(input);
	fclose(output);

	return 0;
}
