#include "celeste.h"
#include "parser.h"
#include "common_ext.h"
#include "celeste_types.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/** 
 * @file
 * Analyzes and passes the argument to function that do computations.
 */

void print_usage(void) {
	int i,j;
	printf("Usage: celeste [options] [filename]\n"
	"Options:\n"
	"  -o           Set output prefix\n"
	"  -d           Set duration (default : 1000) (units admitted: s, m, d, y)\n"
	"  -s,--step    Set step (default : 1) (units admitted: s, m, d, y)\n"
	"  -m <method>  Set numerical method :\n");
	printf("     %s: %s (default)\n", compute_methods[0]->cl_switch[0],
			compute_methods[0]->name[0]);
	for (i = 1; i < NB_METHODS; i++) {
		for (j=0; j<compute_methods[i]->nb_submethods; j++) {
			printf("     %s: %s\n", compute_methods[i]->cl_switch[j], 
					compute_methods[i]->name[j]);
			}
	}
	printf(
	"  -r {b,0}  	Set frame of reference\n"
	"     b: barycenter\n"
	"     0: defined by the file\n"
	"  -w --write	Set write option (chain of the form 0011010)\n"
	"		i-th character:	if 1, the output of the i-th planet will be written\n"
	"				if 0, the output of the i-th planet will not be written\n"
	"  -h,--help	Print this screen\n");
}


/**
 * Translate units.
 *
 * @param s a string containing a number with, possibly, a unit. The unit
 * might be s (second), m (minute), h (hour), d (day) or y (year).
 * 
 * @return a double with number in seconds.
 */
double strunit(char *s) {
	int l = strlen(s);
	double fact = 1;
	switch(s[l-1]) {
		case 'y':
			fact *= 365.24;
		case 'd':
			fact *= 24;
		case 'h':
			fact *= 60;
		case 'm':
			fact *= 60;
		case 's':
			s[l-1] = '\0';
	}
	return fact * strtod(s, NULL);
}

/**
 * Entry point.
 *
 * Analyzes the arguments passed, that consists of a filename and options
 * set by switches. 
 * 
 */
int main(int argc, char *argv[]) {
	int i=1, j, k;
	double duration = 1000;
	double step = 1;
	int method = VELOCITY_VERLET;
	int submethod = 0;	
	int reference = BARYCENTER;
	char *output_pref;
	char* write_option = "";

	output_pref = strdup("./celeste");

	FILE *input = NULL;
	cel_system *sys;
	cel_files *files;

	if (argc==0) {
		print_usage();
		return 0;
	}

	while(i<argc) {
		if(!strcmp("-o", argv[i])) {
			free(output_pref);
			output_pref = strdup(argv[++i]);
			i++;
			continue;
		}

		if(!strcmp("-h", argv[i]) || 
		   !strcmp("--help", argv[i])) {
			print_usage();
			return 0;
		}

		if(!strcmp("-d", argv[i])) {
			duration = strunit(argv[++i]);
			i++;
			continue;
		}
		
		if(!strcmp("-s", argv[i]) ||
		   !strcmp("--step", argv[i])) {
			step = strunit(argv[++i]);
			i++;
			continue;
		}

		if(!strcmp("-r", argv[i])) {
			i++;
			if(strcmp("b", argv[i]) == 0)
				reference = BARYCENTER;
			else if(strcmp("0", argv[i]) == 0)
				reference = ZERO;
			else {
				print_usage();
				return 1;
			}
			i++;
			continue;
		}
			

		if(!strcmp("-m", argv[i])) {
			i++;
			for(j=0; j < NB_METHODS; j++) {
				for(k=0; k<compute_methods[j]->nb_submethods; k++) {
					if(strcmp(compute_methods[j]->cl_switch[k], argv[i]) == 0) {
						break;
					}
				}
				if(k<compute_methods[j]->nb_submethods)
					break;
			}
			if(j == NB_METHODS) {
				print_usage();
				return 1;
			} else {
				method = j;
				submethod = k;
			}
			i++;
			continue;
		}
		if(!strcmp("-w", argv[i]) || !strcmp("--write", argv[i])) {
			write_option = strdup(argv[++i]);
			i++;
			continue;
		}
		if(input != NULL) {
			print_usage();
			return 1;
		}

		input = fopen(argv[i], "r");
		if(input == NULL) {
			fprintf(stderr, "can't open: %s\n", argv[i]);
			return 1;
		}
		

		i++;
	}
	if (input == NULL)
		input = stdin;
	
	parse(input, &sys, step, duration, write_option);
	files = alloc_cel_files(sys, output_pref);

	compute_and_print(sys, files, method, submethod, reference);

	fclose(input);
	free_cel_files(files, sys);
	free_cel_system(sys);
	return 0;
}


