#include "celeste_types.h"
#include <strings.h>
#include <stdlib.h>
#include <math.h>

/** @file
 * Manages basic types.
 * Allocates and free files- and system-related structures.
 */


/**
 * Allocates cel_files structure.
 *
 * Must be freed with free_cel_files.
 * @param sys a celestial system. Used for planet names.
 * @param file_pref the filename prefix.
 */
cel_files *alloc_cel_files(cel_system *sys, char *file_pref) {
	cel_files *files;
	
	files = malloc(sizeof(cel_files));
	files->nb_planets = sys->nb_planets;

#ifndef SPEEDRUN
	char filename[1024];
	int i;

	// Energy
	sprintf(filename, "%s-energy", file_pref);
	files->energy = fopen(filename, "w");
	fprintf(files->energy, "# time\tenergy\n");

	// Angular momentum
	sprintf(filename, "%s-ang_momentum", file_pref);
	files->ang_momentum = fopen(filename, "w");
	fprintf(files->energy, "# time\tang_momentum\n");

	// Planets
	files->planets = calloc(sizeof(FILE*), files->nb_planets);
	for(i = 0; i < files->nb_planets; i++) {
		if(sys->planets[i]->is_to_write) {
			sprintf(filename, "%s-%s", file_pref, sys->planets[i]->name);
			files->planets[i] = fopen(filename, "w");
			fprintf(files->planets[i], "#step=%lf\n",sys->time_step);
		}
	}
#endif // SPEEDRUN

	return files;
		
}

/**
 * Free cel_files allocated with alloc_cel_files.
 */
void free_cel_files(cel_files *files, cel_system *sys) {
#ifndef SPEEDRUN
	int i;
	fclose(files->energy);
	fclose(files->ang_momentum);
	for(i=0; i < files->nb_planets; i++)
		if(sys->planets[i]->is_to_write)
			fclose(files->planets[i]);
	free(files->planets);
	free(files);
#endif // SPEEDRUN
}

/**
 * Allocates cel_system structures.
 *
 * Must be freed with free_cel_system.
 */
cel_system *alloc_cel_system(int nb_planets, int dimension, double step,
		double duration, int *write_options) {
	cel_system *sys;
	int i;
	
	sys = malloc(sizeof(cel_system));
	sys->nb_planets = nb_planets;
	sys->dimension = dimension;

	sys->gr_activated = 0; //GR correction desactivated by default
	sys->mass_fat_body = 0;

	sys->duration = duration;
	sys->time_step = step;

	/* 
	 * We start with _next values, in order to compute acceleration :
	 * time_cur must be initialized with -step.
	 */
	sys->time_cur = -step;
	sys->index_cur = -1;
	sys->index_tot = floor(duration/step);
	sys->total_mass = 0;
	
	sys->refx_cur = calloc(dimension, sizeof(double));
	sys->planets = malloc(sizeof(planet*) * nb_planets);
	for(i=0; i < nb_planets; i++) {
		sys->planets[i] = alloc_planet(dimension);
		sys->planets[i]->is_to_write = write_options[nb_planets-i-1];
	}
	return sys;
}

void free_cel_system(cel_system *sys) {
	int i,j,k;

	for(i=0; i < sys->nb_planets; i++) {
		free_planet(sys->planets[i]);
		if(sys->gr_activated) {
			for(j=0; j<sys->dimension+1; j++) {
				for(k=0; k<sys->dimension+1; k++)
					free(sys->cs[i][j][k]);
				free(sys->cs[i][j]);
			}
			free(sys->cs[i]);
		}
	}
	if(sys->gr_activated)
		free(sys->cs);
	free(sys->refx_cur);
	free(sys->planets);
	free(sys);
	return;
}

/**
 * Allocates a planet.
 */
planet *alloc_planet(int dimension) {
	planet *p;
	
	p = malloc(sizeof(planet));
	p->name = NULL;
	p->mass = 0;
	
	p->x_cur = calloc(dimension, sizeof(double));
	p->v_cur = calloc(dimension, sizeof(double));
	p->a_cur = calloc(dimension, sizeof(double));


	p->x_next = calloc(dimension, sizeof(double));
	p->v_next = calloc(dimension, sizeof(double));
	p->a_next = calloc(dimension, sizeof(double));
	return p;
}

void free_planet(planet *p) {
	free(p->name);
	
	free(p->x_cur);
	free(p->v_cur);
	free(p->a_cur);
	
	free(p->x_next);
	free(p->v_next);
	free(p->a_next);

	free(p);
	return;
}

