#ifndef CELESTE_TYPES_H
#define CELESTE_TYPES_H

#include <stdio.h>

#define BARYCENTER 0
#define ZERO 1

/**
 * File handles.
 *
 * Contains all the file handles necessary to write the output.
 */
typedef struct cel_files {
	FILE *ang_momentum; /**< Angular momentum. */
	FILE *energy;

	int nb_planets;
	FILE **planets;
} cel_files;


/**
 * A planet.
 *
 * Store acceleration, velocity and position.
 * _cur values tracks computations made on _next values.
*/
typedef struct planet {
	char *name;
	int is_to_write;

	double *x_cur;
	double *v_cur;
	double *a_cur;

	double *x_next;
	double *v_next;
	double *a_next;

	double mass;
} planet;

/**
 * Contains the system.
 *
 * Information on energy, total mass, time, angular momentum.
 */
typedef struct cel_system {
	int nb_planets;

	/**
	 * Dimension of the space.
	 *
	 * Should be 2 (plane) or 3 (our space), but can be more.
	 */
	int dimension;

	double total_mass; /**< Used for barycenter. */
	
	/**
	 * For the GR correction
	 */
	int gr_activated; /**< Switch to use the GR correction. */
	double mass_fat_body; /**< Mass of the heaviest body. */
	int index_fat_body; /**< Index of th eheaviest body in the array planets. */
	double ****cs; /**< 4-dimensional array containing the cristoffels symbols (one dimension to take account the N planets, plus three for the coefficients). */
	
	double duration; /**< Duration of simulation. */
	double time_step; /**< Time step between two iterations. */
	double time_cur; /**< Current time. */
	int index_cur; /**< Current step of integration. */
	int index_tot; /**< Number of step of integration. */
	
	double *refx_cur; /**< Position of frame of reference. */
	
	unsigned int gpu_enabled; /**< 1 if and only if GPU is enabled */

	double kin_energy_cur;
	double pot_energy_cur;

	/**
	 * An artefact.
	 * Needed because we compute potential energy and acceleration at the
	 * same time, due to similarities.
	 */
	double pot_energy_next;

	double energy_cur;
	double ang_momentum_cur;
	planet **planets;

} cel_system;

cel_files *alloc_cel_files(cel_system *sys, char *file_pref);
void free_cel_files(cel_files *files, cel_system *sys);

cel_system *alloc_cel_system(int nb_planets, int dimension, double step,
		double duration, int *write_options);
void free_cel_system(cel_system *sys);

planet *alloc_planet();
void free_planet(planet *p);


#endif // CELESTE_TYPES_H
