#include "common_int.h"
#include <stdlib.h>
#include <math.h>
#include "cristoffel.h"

#ifdef SPEEDRUN
#include <time.h>
#endif

#define c 299792458.0
#define G 6.67384E-11

/** @file
 * Computations common to all numerical methods.
 *
 * Computes acceleration, velocity, position, angular momentum and energy.
 * Velocity and position must be provided by an external numerical method.
 * Acceleration, velocity, and position are computed in _next values. All
 * others are computed in _cur values.
 * Prints position for every planet, total energy and total angular momentum
 * of the system.
 * Angular momentum is printed only if the space is a 2d-space.
 */ 

/**
 * Compute current frame of reference.
 *
 * Selected at the first iteration, according to chosen frame of reference.
 */
int (*current_ref) (cel_system *);


/**
 * Prints position of each planet.
 */
int fprint_cur_planets(cel_system *sys, cel_files *files) {
	planet **planets = sys->planets;
	int i, j;

	for(i=0; i < sys->nb_planets; i++) {
		if(planets[i]->is_to_write) {
			for(j = 0; j < sys->dimension; j++)
				fprintf(files->planets[i], "%.15lf\t",
						planets[i]->x_cur[j] - sys->refx_cur[j]);
			fprintf(files->planets[i], "\n");
		}
	}
	return 0;
}

int fprint_cur_energy(cel_system *sys, cel_files *files) {
	double time;

	time = sys->time_cur;
	fprintf(files->energy, "%g\t%g\n", time, sys->kin_energy_cur + sys->pot_energy_cur);
	return 0;
}

int fprint_cur_ang_momentum2d(cel_system *sys, cel_files *files) {
	double time;

	time = sys->time_cur;
	fprintf(files->ang_momentum, "%g\t%g\n", time, sys->ang_momentum_cur);
	return 0;
}

/**
 * Compute each acceleration and total potential energy.
 *
 * However... Potential energy is the *next* potential energy. It is stored
 * in a _next variable (another option could be to store it in a global
 * variable). Not cute.
 */
int compute_a(cel_system *sys, int submethod) {
	int i = 0; int m = 0; 	// ith, mth planet
	int j, k, l;		// jth, kth, lth coordinate
	double *sumx;
	double sqdist, fact, pot_energy;
	double *xi;
	double *rx;
	int dimension = sys->dimension;
	planet **planets = sys->planets;
	
	xi = calloc(dimension, sizeof(double));
	rx = calloc(dimension, sizeof(double));
	sumx = calloc(dimension, sizeof(double));

#ifndef SPEEDRUN
	pot_energy = 0;
#endif // SPEEDRUN
	
	for(i=0; i < sys->nb_planets; i++) {
		for(j=0; j < dimension; j++)
			sumx[j] = 0;
		for(m=0; m < sys->nb_planets; m++) {
			if (i == m || m == sys->index_fat_body)
				continue;
			sqdist = 0;
			for(j=0; j < dimension; j++) {
				rx[j] = planets[m]->x_next[j] - planets[i]->x_next[j];
				sqdist += SQ(rx[j]);
			}

			fact = planets[m]->mass/pow(sqdist, 3.0/2.0);
			
			for(j=0; j < dimension; j++)
				sumx[j] += fact*rx[j];
#ifndef SPEEDRUN
			pot_energy += -planets[m]->mass/sqrt(sqdist);
#endif

		}
		if(sys->gr_activated && i!=sys->index_fat_body) {
			sqdist = 0;
			for(j=0; j<dimension; j++) {
				sqdist += SQ(planets[i]->x_next[j]-planets[sys->index_fat_body]->x_next[j]);
			}
			pot_energy += sys->mass_fat_body/sqrt(sqdist);
		}
#ifndef SPEEDRUN
		pot_energy *= planets[i]->mass;
#endif
	
		if(sys->gr_activated && i!=sys->index_fat_body) {
			//We compute the cristoffel symbols
			double r = 0;
 			for(k=0; k<dimension; k++)
				r += SQ(planets[i]->x_next[k]-planets[sys->index_fat_body]->x_next[k]);
			r = sqrt(r); //distance between the ith planet and the sun
			double rc = 2.0 * G * sys->mass_fat_body / SQ(c); //schwartzchild radius
			sys->cs[i][dimension][dimension][dimension] = 0;
			for(j=0; j<dimension; j++) {
		        	sys->cs[i][dimension][dimension][j] = (planets[i]->x_next[j]-planets[sys->index_fat_body]->x_next[j])/(2*SQ(r)*(r/rc-1));
		        	sys->cs[i][dimension][j][dimension] = (planets[i]->x_next[j]-planets[sys->index_fat_body]->x_next[j])/(2*SQ(r)*(r/rc-1));
			        sys->cs[i][j][dimension][dimension] = (planets[i]->x_next[j]-planets[sys->index_fat_body]->x_next[j])*rc*(1-rc/r)/(2*pow(r,3));
				for(k=0; k<dimension; k++) {
					sys->cs[i][dimension][j][k] = 0;
					sys->cs[i][j][dimension][k] = 0;
					sys->cs[i][j][k][dimension] = 0;
					for(l=0; l<dimension; l++) {
						sys->cs[i][j][k][l] = (planets[i]->x_next[k]-planets[sys->index_fat_body]->x_next[k]) * (planets[i]->x_next[l]-planets[sys->index_fat_body]->x_next[l]) * (3/2*r/rc-1) / (SQ(r) * (1-r/rc));
						if(k==l)
							sys->cs[i][j][k][l] += 1;
						sys->cs[i][j][k][l] *= (planets[i]->x_next[j]-planets[sys->index_fat_body]->x_next[j])*rc/pow(r,3);
					}
				}
			}
		}
		for(j = 0; j < dimension; j++) {
			planets[i]->a_next[j] = G*sumx[j];
			if(sys->gr_activated && i!=sys->index_fat_body) {
				//We compute the correction to a
				for(k=0; k<dimension; k++) {
					for(l=0; l<dimension; l++)
						planets[i]->a_next[j] -= sys->cs[i][j][k][l]*planets[i]->v_next[k]*planets[i]->v_next[l];
					planets[i]->a_next[j] -= 2*sys->cs[i][dimension][dimension][k]*planets[i]->v_next[j]*planets[i]->v_next[k];
				}
				planets[i]->a_next[j] -= sys->cs[i][j][dimension][dimension]*SQ(c);
			}
		}
	}
#ifndef SPEEDRUN
	pot_energy *= G;
	sys->pot_energy_next = pot_energy;
#endif

	free(xi); free(rx); free(sumx);
	return 0;
}

int current_kin_energy(cel_system *sys) {
	int i = 0, j;
	double kin_energy, sqspeed;

	planet **planets = sys->planets;

	kin_energy = 0;

	for(i=0; i < sys->nb_planets; i++) {
		sqspeed = 0;
		for(j = 0; j < sys->dimension; j++)
			sqspeed += SQ(planets[i]->v_cur[j]);
		kin_energy += 0.5 * planets[i]->mass * sqspeed;
	}
	sys->kin_energy_cur = kin_energy;
	return 0;
}

int current_ang_momentum2d(cel_system *sys) {
	int i=0;
	double ang_momentum = 0;

	planet **planets = sys->planets;

	for(i=0; i < sys->nb_planets; i++) {
		ang_momentum += planets[i]->mass*
			(planets[i]->x_cur[0]*planets[i]->v_cur[1]
			- planets[i]->x_cur[1]*planets[i]->v_cur[0]);
	}
	sys->ang_momentum_cur = ang_momentum;
	return 0;
}

/**
 * Compute a frame of reference defined by the barycenter of the system.
 *
 * Can be used in *current_ref
 */
int cur_reference_barycenter(cel_system *sys) {
	int i, k;
	double *barx;
	planet **planets = sys->planets;

	barx = calloc(sys->dimension, sizeof(double));

	for(k=0; k < sys->dimension; k++)
		barx[k] = 0;
	for (i=0; i < sys->nb_planets; i++) {
		for(k=0; k < sys->dimension; k++)
			barx[k] += planets[i]->mass * planets[i]->x_cur[k];
	}
	for(k=0; k < sys->dimension; k++) {
		barx[k] = barx[k]/sys->total_mass;
		sys->refx_cur[k] = barx[k];
	}
	return 0;
}

/**
 * Compute a frame of reference defined by the file.
 *
 * Can be used in *current_ref
 */
int cur_reference_zero(cel_system *sys) {
	int k;
	for(k=0; k < sys->dimension; k++)
		sys->refx_cur[k] = 0;
	return 0;
}

/**
 * Move _next values to _cur ones, computes _cur-dependant values and print
 * them.
 * 
 * Potential energy is moved from _next to _cur.
 * Angular momentum is printed only if the space is a 2d-space.
 */
int next_and_print(cel_system *sys, cel_files *files) {
	int i, k;
	planet **planets = sys->planets;

	
	sys->time_cur += sys->time_step;
	sys->index_cur++;
	for(i = 0; i < sys->nb_planets; i++) {
		for(k=0; k < sys->dimension; k++) {
			planets[i]->x_cur[k] = planets[i]->x_next[k];
			planets[i]->v_cur[k] = planets[i]->v_next[k];
			planets[i]->a_cur[k] = planets[i]->a_next[k];
		}
	}
#ifndef SPEEDRUN
	sys->pot_energy_cur = sys->pot_energy_next;

	(*current_ref)(sys);
	current_kin_energy(sys);
	
	if(sys->dimension == 2) {
		current_ang_momentum2d(sys);
		fprint_cur_ang_momentum2d(sys, files);
	}

	fprint_cur_planets(sys, files);
	fprint_cur_energy(sys, files);
#endif // SPEEDRUN
	return 0;
}

/**
 * Compute and print values, prepare the next iteration.
 * 
 * Compute acceleration, velocity and position, as defined by the external
 * numerical method, in the array compute_methods.
 */
int compute_and_print_next(cel_system *sys, cel_files *files,
		int method, int submethod) {
	int i;
	for(i = 0; i < NB_COMPUTS; i++)
		(*(compute_methods[method]->compute[i]))(sys, submethod);
	next_and_print(sys, files);
	
	return 0;
}

/**
 * Compute and print values.
 *
 * Set frame of reference and print values for the duration asked.
 */
int compute_and_print(cel_system *sys, cel_files *files,
		int method, int submethod, int ref_type) {
#ifdef SPEEDRUN
	struct timespec start_time, end_time;
	double timediff;
#endif // SPEEDRUN
	switch(ref_type) {
		case BARYCENTER:
			current_ref = &cur_reference_barycenter;
			break;
		case ZERO:
			current_ref = &cur_reference_zero;
			break;
	}

#ifdef SPEEDRUN
	fprintf(stderr, "Testing speed of %d iterations over %d planets...\n",
			(int) floor(sys->duration/sys->time_step), sys->nb_planets);
	clock_gettime(CLOCK_REALTIME, &start_time);
#endif // SPEEDRUN
	compute_a(sys, submethod);
	next_and_print(sys, files);
	while(sys->time_cur < sys->duration) {
		compute_and_print_next(sys, files, method, submethod);
	}
#ifdef SPEEDRUN
	clock_gettime(CLOCK_REALTIME, &end_time);
	timediff = difftime(end_time.tv_sec, start_time.tv_sec) +
		1E-9 * (double) (end_time.tv_nsec - start_time.tv_sec);
	fprintf(stderr, "Done in %lg sec.\n", timediff);
#endif // SPEEDRUN


	return 0;
}
