#include "explicit_rk.h"
#include "celeste_types.h"
#include <math.h>
#include <stdlib.h>
#include "rk_matrices.h"
#include "cristoffel.h"

#define G 6.67384E-11


int compute_x_and_v_explicit_rk(cel_system *sys, int submethod) {
	planet **planets = sys->planets;
	int dimension = sys->dimension;
	int nb_planets = sys->nb_planets;
	int i; //ith planet
	int j; //jth coordinate

	// Allocation of temporary matrices if t==0
	static double ***K_pos, ***K_vel, ***pos, ***vel; //matrix of the form A[current_step][planet_i][coord_j]
	int nb_stages = rk[submethod].nb_stages;
	int stage_cur;
	int stage_prev;
	if(sys->index_cur==0) {
		alloc_rk_matrices(&K_pos, &K_vel, &pos, &vel, nb_stages, sys);
	}

	//initialization of Y[0] and K[0] (stage_cur = 0)
	for(i=0; i<nb_planets; i++) {
		for(j=0; j<dimension; j++) {
			pos[0][i][j]=planets[i]->x_cur[j];
			vel[0][i][j]=planets[i]->v_cur[j];
		}
	}
	compute_K_explicit_rk(K_pos[0], K_vel[0], pos[0], vel[0], sys);
	if(sys->gr_activated) {
		compute_cristoffel_symbols(pos[0],sys);
		compute_K_gr_correction(K_vel[0], vel[0], sys);
	}

	//explicit runge-kutta method
	for(stage_cur=1; stage_cur<nb_stages+1; stage_cur++) {
		for(i=0; i<nb_planets; i++) {
			for(j=0; j<dimension; j++) {
				pos[stage_cur][i][j] = pos[0][i][j];
				vel[stage_cur][i][j] = vel[0][i][j];
				for(stage_prev=0; stage_prev<stage_cur; stage_prev++) {
					pos[stage_cur][i][j] += rk[submethod].A[stage_cur * nb_stages + stage_prev] * K_pos[stage_prev][i][j];
					vel[stage_cur][i][j] += rk[submethod].A[stage_cur * nb_stages + stage_prev] * K_vel[stage_prev][i][j];
				}
			}
		}
		if(stage_cur<nb_stages) {
			compute_K_explicit_rk(K_pos[stage_cur], K_vel[stage_cur], pos[stage_cur], vel[stage_cur], sys);
			if(sys->gr_activated) {
				compute_cristoffel_symbols(pos[stage_cur], sys);
				compute_K_gr_correction(K_vel[stage_cur], vel[stage_cur], sys);
			}
		}
	}
	

	//saving of results in sys
	for(i=0; i<nb_planets; i++) {
		for(j=0; j<dimension; j++) {
			planets[i]->x_next[j] = pos[nb_stages][i][j];
			planets[i]->v_next[j] = vel[nb_stages][i][j];
		}
	}

	//free memory if t==tf-step
	if(sys->index_cur==sys->index_tot) {
		free_rk_matrices(&K_pos, &K_vel, &pos, &vel, nb_stages, sys);
	}

	return 0;
}

//Do nothing because all the work is done by compute_x_and_v
int compute_nothing_explicit_rk(cel_system *sys, int submethod) {
	return 0;
}

void compute_K_explicit_rk(double **K_pos, double **K_vel, double **pos, double **vel, cel_system *sys) {
	planet **planets = sys->planets;
	int i,k; 	//ith or kth planets
	int j; 		//jth coordinates
	int nb_planets = sys->nb_planets;
	int dimension = sys->dimension;

	double sqdist;
	double time_step = sys->time_step;
	for(i=0; i<nb_planets; i++) {
		for(j=0; j<dimension; j++) {
			K_pos[i][j] = time_step * vel[i][j];
			K_vel[i][j] = 0;
		}
	}
	for(i=0; i<nb_planets; i++) {
		for(k=0;k<nb_planets;k++) {
			if(i==k || k==sys->index_fat_body) {
				continue;
			}
			else {
				sqdist = 0;
				for(j=0; j<dimension; j++) {
					sqdist += SQ( pos[i][j] - pos[k][j] );
				}
				for(j=0; j<dimension; j++) {
					K_vel[i][j] -= time_step * G * planets[k]->mass * (pos[i][j]-pos[k][j]) / pow(sqdist,3.0/2);
				}
			}
		}
	}
}

void alloc_rk_matrices(double ****K_pos, double ****K_vel, double ****pos, double ****vel, int nb_stages, cel_system *sys) {
	int stage_cur;
	int nb_planets = sys->nb_planets;
	int i; 	//ith planet
	int dimension = sys->dimension;

	//allocation of memory
	(*pos) = malloc( (nb_stages+1) * sizeof(double**) );
	(*vel) = malloc( (nb_stages+1) * sizeof(double**) );
	(*K_pos) = malloc( nb_stages * sizeof(double**) );
	(*K_vel) = malloc( nb_stages * sizeof(double**) );
	for(stage_cur=0; stage_cur<nb_stages; stage_cur++) {
		(*pos)[stage_cur] = malloc( nb_planets * sizeof(double*) );
		(*vel)[stage_cur] = malloc( nb_planets * sizeof(double*) );
		(*K_pos)[stage_cur] = malloc( nb_planets * sizeof(double*) );
		(*K_vel)[stage_cur] = malloc( nb_planets * sizeof(double*) );
		for(i=0; i<nb_planets; i++) {
			(*pos)[stage_cur][i] = malloc( dimension * sizeof(double) );
			(*vel)[stage_cur][i] = malloc( dimension * sizeof(double) );
			(*K_pos)[stage_cur][i] = malloc( dimension * sizeof(double) );
			(*K_vel)[stage_cur][i] = malloc( dimension * sizeof(double) );
		}
	}
	(*pos)[nb_stages] = malloc( nb_planets * sizeof(double*) );
	(*vel)[nb_stages] = malloc( nb_planets * sizeof(double*) );
	for(i=0; i<nb_planets; i++) {
		(*pos)[nb_stages][i] = malloc( dimension * sizeof(double) );
		(*vel)[nb_stages][i] = malloc( dimension * sizeof(double) );
	}
}

void free_rk_matrices(double ****K_pos, double ****K_vel, double ****pos, double ****vel, int nb_stages, cel_system *sys) {
	int stage_cur;
	int nb_planets = sys->nb_planets;
	int i;
	for(stage_cur=0; stage_cur<nb_stages; stage_cur++) {
		for(i=0; i<nb_planets; i++) {
			free( (*pos)[stage_cur][i] );
			free( (*vel)[stage_cur][i] );
			free( (*K_pos)[stage_cur][i] );
			free( (*K_vel)[stage_cur][i] );
		}
		free( (*pos)[stage_cur] );
		free( (*vel)[stage_cur] );
		free( (*K_pos)[stage_cur] );
		free( (*K_vel)[stage_cur] );
	}
	for(i=0; i<nb_planets; i++) {
		free( (*pos)[nb_stages][i] );
		free( (*vel)[nb_stages][i] );
	}
	free( (*pos)[nb_stages] );
	free( (*vel)[nb_stages] );
	
	free(*pos);
	free(*vel);
	free(*K_pos);
	free(*K_vel);
}
