%{
/** @file
 * Parses .ini files.
 * 
 * Parsing is make in two passes:
 * 1. The bison-part is a quite generic parser analyze the .ini grammar. 
 * Each section contains a list of keys and a head. Sections are pushed in
 * a list.
 * 2. A specific function fills data structures planet and cel_system.
 */

#include <stdlib.h>
#include <math.h>
#include "mylib.h"
#include "parsertypes.h"
#include "parser.h"
#include "yy.lex.h"
#include <stdio.h>
#include <string.h>
#include "celeste_types.h"


int yyerror(const char *p) {
	fprintf(stderr, "%s\n", p);
	exit(EXIT_FAILURE);
}

slist *file_sections;

%}

%union {
	double	dval;
	int		ival;
	char	*str;
	key		*key;
	section	*section;
	slist	*list;
};

%token <str> STRING
%token <ival> INT
%token <dval> DBL
%token EQ BRA KET NEWLINE UNDERSCORE
%type <dval> VAL
%type <key> KEY
%type <str> HEAD
%type <section> SECTION
%type <list> KEYS SECTIONS
%start input
%%
input	: SECTIONS { file_sections = $1; }
	  	;

SECTIONS: /* empty */ { $$ = slist_new(); }
		| SECTIONS SECTION { $$ = $1; slist_push($$, $2);}
		;

SECTION	: HEAD KEYS { $$ = section_new(); $$->head = $1; $$->keys = $2; }
		;

HEAD	: BRA STRING KET NEWLINE { $$ = strdup($2); }
		;

KEYS	: /* empty */ { $$ = slist_new(); }
		| KEYS KEY { $$ = $1; slist_push($$, $2); }
		;

KEY		: STRING EQ VAL NEWLINE 
	 		{ $$ = key_new(); $$->name = strdup($1);
			  $$->val = $3; }
		| STRING UNDERSCORE INT EQ VAL NEWLINE
			{ $$ = key_new(); $$->name = strdup($1);
			  $$->val = $5; $$->subscript = $3; }
		;

VAL		: DBL { $$ = $1; }
	 	| INT { $$ = (double) $1; }
		;

%%

int get_dimension(slist *sections) {
	section *sec;
	slist_rewind(sections);
	key *k;
	int gr_found,dimension;
	while((sec = (section*) slist_cur(sections)) != NULL) {
		gr_found=0;
		slist_rewind(sec->keys);
		while((k = (key*) slist_cur(sec->keys)) != NULL) {
	        	if(!strcmp("gractivated",k->name) && (k->val == 1)) {
				gr_found=1;
				break;
			}
			slist_next(sec->keys);
		}
		if(gr_found==0) {
			dimension = (slist_length(sec->keys)-1)/2;
			break;
		}
		slist_next(sections);
	}
	return dimension;
}


int verify_sections(slist *sections) {
	int nb_sections = slist_length(sections);
	int i, j;
	char **heads;
	heads = calloc(nb_sections, sizeof(char*));
	slist_rewind(sections);
	section *sec;
	
	i = 0;
	while((sec = (section*) slist_cur(sections)) != NULL) {
		// Two sections must have a different head.
		heads[i] = strdup(sec->head);
		for(j = 0; j < i-1; j++) {
			if(!strcmp(heads[i],heads[j])) {
				fprintf(stderr, "Two sections named %s.\n", heads[i]);
				exit(EXIT_FAILURE);
			}
		}
		i++;
		slist_next(sections);
	}
	for(i = 0; i < nb_sections; i++)
		free(heads[i]);
	free(heads);
	
	return 0;
}

/*
 * Return 1 if the GR correction is activated, else return 0
 */
int get_gr_activated(slist* sections) {
	section *sec;
	slist_rewind(sections);
	key *k;
	int gr_found=0;
	while((sec = (section*) slist_cur(sections)) != NULL) {
		slist_rewind(sec->keys);
		while((k = (key*) slist_cur(sec->keys)) != NULL) {
			if(!strcmp("gractivated",k->name) && (k->val == 1)) {
				if(gr_found) {
					fprintf(stderr, "two keys %s: only one fat body is possible\n", k->name);
					exit(EXIT_FAILURE);
				}
				gr_found=1;
			}
			slist_next(sec->keys);
		}
		slist_next(sections);
	}
	if(gr_found)
		return 1;
	else
		return 0;
}

/**
 * Fills data structures planet and cel_system with .ini data.
 * 
 * Allocation of sys is made here.
 */
int parse(FILE *input_file, cel_system **sys, double step, double duration, char *write_options) {
	int nb_planets, i, j, l;
	planet **planets;
	section *sec;
	slist *keys;
	key *k;
	int dimension;
	int nb_keys;
	
	int *xs; // Array. xs[i] = 1 if we found x_[i].
	int *vs; // Array. vs[i] = 1 if we found v_[i].
	int mass_found;
	int gr_found = 0;

	yyin = input_file;

	yyparse();
	
	verify_sections(file_sections);
	gr_found = get_gr_activated(file_sections);
	dimension = get_dimension(file_sections);
	if(dimension == 0) {
		fprintf(stderr, "dimension is 0\n");
		exit(EXIT_FAILURE);
	}

	xs = calloc(dimension, sizeof(int));
	vs = calloc(dimension, sizeof(int));

	nb_planets = slist_length(file_sections);
	int* to_write = malloc(nb_planets*sizeof(int));
	if(!strcmp(write_options,"")) {
		for(i=0; i<nb_planets; i++) {
			to_write[i] = 1;
		}
	}
	else if(strlen(write_options) != nb_planets) {
		fprintf(stderr, "error: wrong chain length for the -write options\n");
		free(to_write);
		exit(EXIT_FAILURE);
	}
	else {
		for(i=0; i<nb_planets; i++) {
			if(write_options[i] != '1' && write_options[i] != '0') {
				fprintf(stderr, "error: chain[%d] different frome 0 or 1 in --write %s\n", i, write_options);
				free(to_write);
				exit(EXIT_FAILURE);
			}
			to_write[i] = write_options[i]-'0';
		}
	}		
	*sys = alloc_cel_system(nb_planets, dimension, step, duration, to_write);
	if(gr_found) {
		(*sys)->gr_activated = 1;
		(*sys)->cs = malloc(nb_planets*sizeof(double***));
		for(i=0; i<nb_planets; i++) {	
			(*sys)->cs[i] = malloc((dimension+1)*sizeof(double**));
			for(j=0; j<dimension+1; j++) {
				(*sys)->cs[i][j] = malloc((dimension+1)*sizeof(double*));
				for(l=0; l<dimension+1; l++) {
					(*sys)->cs[i][j][l] = calloc((dimension+1),sizeof(double));
				}
			}
		}
	}

	planets = (*sys)->planets;
	
	i=0;
	while((sec = (section*) slist_pop(file_sections)) != NULL) {
		memset(xs, 0, sizeof(int)*dimension);
		memset(vs, 0, sizeof(int)*dimension);
		mass_found = 0;
		gr_found = 0;
		planets[i]->name = strdup(sec->head);
		keys = sec->keys;
		nb_keys = slist_length(sec->keys);
		if((nb_keys != 2*dimension+1) && (nb_keys != 2) && (nb_keys != 2*dimension+2)) {
			fprintf(stderr, "wrong number of keys in section %s\n", sec->head);
			exit(EXIT_FAILURE);
		}
		
		// Uniqueness of each key is checked.
		// Subscript is also checked for x and v.
		while((k = (key*)slist_pop(keys)) != NULL) {
			if(!strcmp("mass",k->name)) {
				if(mass_found) {
					fprintf(stderr, "two keys %s in section %s\n", k->name, sec->head);
					exit(EXIT_FAILURE);
				}
				mass_found = 1;
				if(gr_found) {
					(*sys)->mass_fat_body = k->val;
					(*sys)->index_fat_body = i;
				}
				planets[i]->mass = k->val;
				(*sys)->total_mass += k->val;
			} else if(!strcmp("gractivated",k->name)) {
				if(k->val == 1) {
					if(mass_found) {
						(*sys)->mass_fat_body = planets[i]->mass;
						(*sys)->index_fat_body = i;
					}
					gr_found = 1;
				}
			} else if(!strcmp("x",k->name)) {
				if(k->subscript == -1) {
					if(xs[0]) {
						fprintf(stderr, "two keys %s in section %s\n",
							k->name, sec->head);
						exit(EXIT_FAILURE);
					}
					xs[0] = 1;
					planets[i]->x_next[0] = k->val;
				} else {
					if(k->subscript >= dimension) {
						fprintf(stderr,
							"subscript too high for %s_%d in section %s\n",
							k->name, k->subscript, sec->head);
						exit(EXIT_FAILURE);
					}
					if(xs[k->subscript]) {
						fprintf(stderr, "two keys %s_%d in section %s\n",
							k->name, k->subscript, sec->head);
						exit(EXIT_FAILURE);
					}
					xs[k->subscript] = 1;
					planets[i]->x_next[k->subscript] = k->val;
				}
			} else if(!strcmp("y",k->name)) {
				if(dimension < 2) {
					fprintf(stderr,
						"subscript %d is too high for %s in section %s\n",
						k->subscript, k->name, sec->head);
					exit(EXIT_FAILURE);
				}
				if(xs[1]) {
					fprintf(stderr, "two keys %s in section %s\n",
						k->name, sec->head);
					exit(EXIT_FAILURE);
				}
				xs[1] = 1;	
				planets[i]->x_next[1] = k->val;
			} else if(!strcmp("vx",k->name)) {
				if(vs[0]) {
					fprintf(stderr, "two keys %s in section %s\n",
						k->name, sec->head);
					exit(EXIT_FAILURE);
				}
				vs[0] = 1;
				planets[i]->v_next[0] = k->val;
			} else if(!strcmp("vy",k->name)) {
				if(dimension < 2) {
					fprintf(stderr,
						"subscript too high for %s in section %s\n",
						k->name, sec->head);
					exit(EXIT_FAILURE);
				}
				if(vs[1]) {
					fprintf(stderr, "two keys %s in section %s\n",
						k->name, sec->head);
					exit(EXIT_FAILURE);
				}
				vs[1] = 1;	
				planets[i]->v_next[1] = k->val;
			} else if(!strcmp("v", k->name)) {
				if(k->subscript >= dimension) {
					fprintf(stderr,
						"subscript %d is too high for %s in section %s\n",
						k->subscript, k->name, sec->head);
					exit(EXIT_FAILURE);
				}
				if(vs[k->subscript]) {
					fprintf(stderr, "two keys %s in section %s\n",
						k->name, sec->head);
					exit(EXIT_FAILURE);
				}
				vs[k->subscript] = 1;
				planets[i]->v_next[k->subscript] = k->val;
			} else {
				fprintf(stderr, "key name unrecognized: %s\n", k->name);
				exit(EXIT_FAILURE);
			}
		}
		i++;
	}
	slist_free(file_sections, (slist_destruct_data) section_free);
	
	return 0;
}
