/*
 * Create various distribution of stars.
 *
 * The goal is to be allowed to generate a lot of initial configuration.
 */
#include "random_coord.h"
#include "galaxy_types.h"
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#define SQ(x) ((x)*(x))

/*
 * Allocate a gal_system.
 *
 * Set parameter dimension, nb_element, duration and nbstep.
 * nb_element might be change using realloc_gal_system.
 */
gal_system *alloc_gal_system(uint dimension, uint nb_element,
		double duration, uint nbstep) {
	gal_system *sys;
	sys = malloc(sizeof(gal_system));

	sys->dimension = dimension;
	sys->nbstep = nbstep;
	sys->nb_element = nb_element;
	
	sys->duration = duration;
	sys->stepsize = duration/nbstep;

	sys->vel = calloc(dimension*nb_element,sizeof(double));
	sys->pos = calloc(dimension*nb_element,sizeof(double));
	sys->mass = calloc(nb_element,sizeof(double));

	return sys;
}

/*
 * Change and reallocate the number of stars in a gal_system.
 */
int realloc_gal_system(gal_system *sys, uint nb_element) {
	uint dimension = sys->dimension;
	if(nb_element > sys->nb_element) {
		sys->vel = realloc(sys->vel,dimension*nb_element*sizeof(double));
		sys->pos = realloc(sys->pos, dimension*nb_element*sizeof(double));
		sys->mass = realloc(sys->mass, nb_element*sizeof(double));
		sys->nb_element = nb_element;
	}
	return 0;
}

double vect_length(uint dim, double *vect) {
	uint i;
	double sqlength = 0;
	for(i = 0; i < dim; i++) 
		sqlength += SQ(vect[i]);
	return sqrt(sqlength);
}

/*
 * Create a spherical galaxy centered on center_pos, with a mean velocity
 * equals to mean_vel. If either center_pos or mean_vel is NULL, it will be
 * considered as zero.
 * The galaxy is placed between start_at and stop_at indexes of sys->vel,
 * sys->pos and sys->mass.
 */
int create_spherical_galaxy(gal_system *sys, double typical_length,
		double radius_max, double total_mass, uint start_at, uint stop_at) {
	
	double mass_element = total_mass / (stop_at-start_at+1);
	uint dim = sys->dimension;

	if(dim != 3) {
		fprintf(stderr, "create_spherical_galaxy work only for 3 dimensions.\n");
		abort();
	}

	uint planet;
	double pos_norm, phi, theta, vel_norm;
	for(planet=start_at; planet <= stop_at; planet++) {
		pos_norm = random_hernquist_radius(typical_length, radius_max);
		phi = random_plain_phi();
		theta = random_plain_theta();

		sys->pos[planet*dim+X] = pos_norm * sin(theta) * cos(phi);
		sys->pos[planet*dim+Y] = pos_norm * sin(theta) * sin(phi);
		sys->pos[planet*dim+Z] = pos_norm * cos(theta);

		vel_norm = random_hernquist_velocity_norm(pos_norm, typical_length, total_mass);
		phi = random_plain_phi();
		theta = random_plain_theta();
		
		sys->vel[planet*dim+X] = vel_norm * sin(theta) * cos(phi);
		sys->vel[planet*dim+Y] = vel_norm * sin(theta) * sin(phi);
		sys->vel[planet*dim+Z] = vel_norm * cos(theta);

		sys->mass[planet] = mass_element;
	}
	
	return 0;
}

/*
 * Create a uniform gas in a spherical box with a Boltzmann statistic for
 * velocities.
 */
int create_uniform_gas(gal_system *sys, double typical_velocity,
		double radius, double total_mass, uint start_at, uint stop_at) {
	double mass_element = total_mass / (stop_at-start_at+1);
	uint dim = sys->dimension;

	if(dim != 3) {
		fprintf(stderr, "create_uniform_gas work only for 3 dimensions.\n");
		abort();
	}


	uint planet;
	
	double vel_norm, phi, theta, pos_norm;
	for(planet=start_at; planet <= stop_at; planet++) {
		pos_norm = random_plain_norm(radius);
		phi = random_plain_phi();
		theta = random_plain_theta();
		sys->pos[planet*dim+X] = pos_norm * sin(theta) * cos(phi);
		sys->pos[planet*dim+Y] = pos_norm * sin(theta) * sin(phi);
		sys->pos[planet*dim+Z] = pos_norm * cos(theta);

		vel_norm = random_boltzmann_norm(typical_velocity);
		phi = random_plain_phi();
		theta = random_plain_theta();
		
		sys->vel[planet*dim+X] = vel_norm * sin(theta) * cos(phi);
		sys->vel[planet*dim+Y] = vel_norm * sin(theta) * sin(phi);
		sys->vel[planet*dim+Z] = vel_norm * cos(theta);

		sys->mass[planet] = mass_element;
	}

	return 0;
	
}

/*
 * Create a single particle.
 */
int create_particle(gal_system *sys, double *position, double *velocity,
		double mass, uint at) {
	uint dim = sys->dimension;
	uint k;
	for(k = 0; k < dim; k++) {
		sys->pos[at*dim+k] = position[k];
		sys->vel[at*dim+k] = velocity[k];
		sys->mass[at] = mass;
	}
	return 0;
}

/*
 * Add a position to a part of a gal_system.
 *
 * Each star between start_at and stop_at (inclusive) are shifted through the
 * *position vector.
 */
int add_position(gal_system *sys, double *position, uint start_at, uint stop_at) {
	uint planet;
	uint dim = sys->dimension;
	uint k;
	for(planet = start_at; planet <= stop_at; planet++) {
		for(k = 0; k < dim; k++)
			sys->pos[planet*dim+k] += position[k];
	}
	return 0;
}

/*
 * Add a velocity to a part of a gal_system.
 *
 * Each star between start_at and stop_at (inclusive) has its velocity added with
 * the *velocity vector.
 */
int add_velocity(gal_system *sys, double *velocity, uint start_at, uint stop_at) {
	uint planet;
	uint dim = sys->dimension;
	uint k;
	for(planet = start_at; planet <= stop_at; planet++) {
		for(k = 0; k < dim; k++)
			sys->vel[planet*dim+k] += velocity[k];
	}
	return 0;
}

/*
 * Add angular momentum to a part of a gal_system.
 *
 * Each star between start_at and stop_at (inclusive) receive angular momentum
 * with an exponential distribution.
 * (experimental)
 */
int add_ang_momentum(gal_system *sys, double max_speed, double typical_radius,
		uint start_at, uint stop_at) {
	uint planet;
	uint dim = sys->dimension;
	double dist_from_center;
	double plane_norm;
	double vel_norm;
	double vel[2];
	for(planet = start_at; planet <= stop_at; planet++) {
		dist_from_center = vect_length(dim, &sys->pos[planet*dim]);
		plane_norm = vect_length(2, &sys->pos[planet*dim]);
		vel_norm = max_speed*exp(-dist_from_center/typical_radius);


		vel[X] = sys->pos[planet*dim+Y]/plane_norm*vel_norm;
		vel[Y] = -sys->pos[planet*dim+X]/plane_norm*vel_norm;
		sys->vel[planet*dim+X] += vel[X];
		sys->vel[planet*dim+Y] += vel[Y];

	}
	return 0;
}


/*
 * Free the memory of a gal_system.
 */
void free_gal_system(gal_system *sys) {
	free(sys->pos);
	free(sys->vel);
	free(sys->mass);
	free(sys);
}
