#include "gpu.h"
#include "galaxy_types.h"
#include "parser.h"
#include <sys/stat.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include <getopt.h>

#define G 6.67398E-11


enum { EULER, VERLET };

void print_usage(void) {
	printf(
			"Usage: gepuku [options] [file]\n"
        	"Options:\n"
			"  -o, --output           set output directory\n"
			"  -w, --write-every      write every n iterations (default: 1)\n"
			"  -m, --method           choose method of integration:\n"
			"                         e: euler\n"
			"                         v: verlet (default)\n"
			"  -h, --help             print this screen\n"
		);
}
static int help = 0;

/*
 * Executes a recursive directory creation to path "dir".
 * It allows to create a directory and multiple subdirectories in order to make
 * the adress "dir" writable.
 */
int recursive_mkdir(const char *dir) {
	char *p, *npath;
	npath = strdup(dir);    /* So we can write to it. */
	struct stat sb;
	mode_t mode = S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH;
	
	/* Check whether or not we need to do anything with intermediate dirs. */
	
	/* Skip leading slashes. */
	p = npath;
	while(*p == '/')
	p++;

	while((p = strchr (p, '/'))) {
		*p = '\0';
		if (stat (npath, &sb) != 0) {
			if (mkdir (npath, mode)) {
				fprintf(stderr, "cannot create directory `%s': %s\n", npath, strerror(errno));
				free(npath);
				return 1;
			}
		} else if (S_ISDIR (sb.st_mode) == 0) {
			fprintf(stderr, "`%s': file exists but is not a directory\n", npath);
			free(npath);
			return 1;
		}

		*p++ = '/';   /* restore slash */
		while (*p == '/')
			p++;
	}

	/* Create the final directory component. */
	if (stat (npath, &sb) && mkdir (npath, mode)) {
		fprintf(stderr, "cannot create directory `%s': %s\n", npath, strerror (errno));
		free(npath);
		return 1;
	}

	free(npath);
	return 0;
}

/*
 * Parses the options, and passes them to the computation.
 * The switches are:
 *  -o, --output: the output directory
 *  -w, --write-every: write every n iteration
 *  -m, --method: select method (Euler or Verlet)
 *  -h, --help: calls print_usage() and exit
 * It also expect a filename. The file should describe how the bodies are
 * distributed. If no filename is given, the input is set to stdin.
 */
int main (int argc, char *argv[]) {
	
	int write_every = 1;
	int method = VERLET;
	
	char *output_directory = strdup("./galaxy");

	int c;

	// This loop parses each switch
	while (1) {
		static struct option long_options[] = {
			{"output",		required_argument,	0, 'o'},
			{"write-every",	required_argument,	0, 'w'},
			{"method",		required_argument,	0, 'm'},
			{"help",		no_argument,		&help, 1},
			{0,				0,					0,  0 }
		};
		int option_index = 0;

		c = getopt_long(argc, argv, "o:w:h", long_options, &option_index);

		if (c == -1)
			break;

		switch(c) {
			case 0:
				break;
			case 'h':
				help = 1;
				break;
			case 'o':
				output_directory = optarg;
				break;
			case 'w':
				write_every = atoi(optarg);
				break;
			case 'm':
				if(optarg[1] == '\0') {
					if(optarg[0] == 'e') {
						method = EULER;
						break;
					}
					if(optarg[0] == 'v') {
						method = VERLET;
						break;
					}
				}
			case '?': // Unknown switch
				print_usage();
				return EXIT_FAILURE;
			default: // Error
				abort();
		}
	}
	if(help) {
		print_usage();
		return EXIT_SUCCESS;
	}
	if(recursive_mkdir(output_directory))
		abort();

	char *fn;
	// Setting filename
	if (optind == argc) // No filename supplied
		fn = NULL;
	else if(optind+1 == argc) // One argument. Must be the filename.
		fn = argv[optind];
	else { // More than one argument. Must be an error.
		print_usage();
		return EXIT_FAILURE;
	}
	

	
	gal_system *sys;
	sys = gal_system_from_file(fn);
	switch(method) {
		case EULER:
			compute_euler_gpu (sys->nb_element, sys->dimension, sys->nbstep, sys->stepsize,
					(double (*)[]) sys->pos, (double (*)[]) sys->vel,
					sys->mass, output_directory, write_every);
			break;
		case VERLET:
			compute_verlet_gpu (sys->nb_element, sys->dimension, sys->nbstep, sys->stepsize,
					(double (*)[]) sys->pos, (double (*)[])sys->vel,
					sys->mass, output_directory, write_every);
			break;
		default:
			abort();
	}

	return 0;
}



