/*
 * Sets up the stellar system.
 *
 * After calling the generic .ini parser, we fill, for each section, a set of
 * stars according to the parsed variables. 
 */

#include "parser.h"
#include "parsertypes.h"
#include "galaxy_types.h"
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#pragma GCC diagnostic ignored "-Wmaybe-uninitialized"

#define DIM 3

/*
 * For each value _val, we set up a bool variable prefixed by _has.
 * This variable is 0 if we haven't set the variable _val.
 * Otherwise, it is 1.
 */
#define INIT_VALUE(_val) bool _has_##_val = 0

/*
 * Returns 1 if _val has been set. 0 otherwise.
 */
#define HAS_VALUE(_val) _has_##_val

/*
 * Set the scalar value _val with _key.
 * _key must be a char pointer.
 * _val can be either an integer or a float. The conversion of _key is done
 * accordingly.
 */
#define ADD_SCALAR_VALUE(_val,_key)                                         \
	do {                                                                    \
		char *_endptr;                                                      \
		if(_has_##_val) {                                                   \
			fprintf(stderr, "Two keys named " #_val " in %s.\n", filename); \
			abort();                                                        \
		}                                                                   \
		_has_##_val = 1;                                                    \
		if((typeof(_val)) 1.1 == 1) /* integer */                           \
			_val = (typeof(_val)) strtol(_key->val, &_endptr, 10);          \
		else /* float */                                                    \
			_val = (typeof(_val)) strtod(_key->val, &_endptr);              \
		if(_endptr[0] != '\0') {                                            \
			fprintf(stderr, "Key %s: syntax error.\n", _key->name);         \
			abort();                                                        \
		}                                                                   \
		key_free(_key);                                                     \
	} while(0)


/*
 * Set the scalar value _val with _key.
 * _key must be a char pointer. _key must contain 3 floats separated by
 * commas. Ex: "23.5,53E-6,-36"
 * _val must be an array whose dimension is DIM.
 */
#define ADD_VECTOR_VALUE(_val,_key)                                         \
	do {                                                                    \
		char *_nptr = _key->val;                                            \
		char *_endptr;                                                      \
		int _i;                                                             \
		if(_has_##_val) {                                                   \
			fprintf(stderr, "Two keys named " #_val " in %s.\n", filename); \
			abort();                                                        \
		}                                                                   \
		_has_##_val = 1;                                                    \
		for(_i = 0; _i < DIM; _i++) {                                       \
			_val[_i] = (typeof(_val[_i])) strtod(_nptr, &_endptr);          \
			if(_endptr[0] == '\0')                                          \
				break;                                                      \
			if(_endptr[0] != ',') {                                         \
				fprintf(stderr, "Key %s: syntax error.\n", _key->name);     \
				abort();                                                    \
			}                                                               \
			_nptr = _endptr+1;                                              \
		}                                                                   \
		for(; _i < DIM; _i++)                                               \
			_val[_i] = 0;                                                   \
		if(_endptr[0] != '\0') {                                            \
			fprintf(stderr, "Key %s: syntax error.\n", _key->name);         \
			abort();                                                        \
		}                                                                   \
		key_free(_key);                                                     \
	} while(0)

/*
 * If _val has not been set, exit program.
 */
#define CHECK_VALUE(_val)                                                   \
	do {                                                                    \
		if(!_has_##_val) {                                                  \
			fprintf(stderr, "No key " #_val " in %s.\n", filename);         \
			abort();                                                        \
		}                                                                   \
	} while(0)


/*
 * Parse the [global] section.
 *
 * Allocate the system with good variables: the duration and the number of step.
 * The number of star is set to 0. Each section will resize the global system, in
 * order to put its stars.
 */
gal_system *parse_global_section(section *global_section, char *filename) {
	dlist *keys = global_section->keys;

	key *current_key;
	gal_system *sys;

	double duration;
	uint nb_step;
	uint dimension = DIM;
	uint nb_element = 0;

	INIT_VALUE(duration);
	INIT_VALUE(nb_step);
	while((current_key = dlist_pop(keys)) != NULL) {
		if(!strcmp(current_key->name, "duration"))
			ADD_SCALAR_VALUE(duration, current_key);
		else if(!strcmp(current_key->name, "nb_step"))
			ADD_SCALAR_VALUE(nb_step, current_key);
		else {
			fprintf(stderr, "Unexpected key %s in %s.\n", current_key->name, filename);
			abort();
		}
	}
	CHECK_VALUE(duration);
	CHECK_VALUE(nb_step);

	sys = alloc_gal_system(dimension, nb_element, duration, nb_step);
	return sys;
}

/*
 * Parse the [hernquist_galaxy] section.
 *
 * It has the following keys:
 * - typical_length
 * - radius_max
 * - total_mass
 * - start_at
 * - stop_at
 * - center_position (optional)
 * - mean_velocity (optional)
 */
int parse_hernquist_galaxy_section(gal_system *sys, section *sec, char *filename) {
	dlist *keys = sec->keys;

	key *current_key;
	double typical_length;
	double radius_max;
	double total_mass;
	uint start_at;
	uint stop_at;
	double center_position[DIM];
	double mean_velocity[DIM];

	INIT_VALUE(typical_length);
	INIT_VALUE(radius_max);
	INIT_VALUE(total_mass);
	INIT_VALUE(start_at);
	INIT_VALUE(stop_at);
	INIT_VALUE(center_position);
	INIT_VALUE(mean_velocity);
	while((current_key = dlist_pop(keys)) != NULL) {
		if(!strcmp(current_key->name, "typical_length"))
			ADD_SCALAR_VALUE(typical_length, current_key);
		else if(!strcmp(current_key->name, "radius_max"))
			ADD_SCALAR_VALUE(radius_max, current_key);
		else if(!strcmp(current_key->name, "total_mass"))
			ADD_SCALAR_VALUE(total_mass, current_key);
		else if(!strcmp(current_key->name, "start_at"))
			ADD_SCALAR_VALUE(start_at, current_key);
		else if(!strcmp(current_key->name, "stop_at"))
			ADD_SCALAR_VALUE(stop_at, current_key);
		else if(!strcmp(current_key->name, "center_position"))
			ADD_VECTOR_VALUE(center_position, current_key);
		else if(!strcmp(current_key->name, "mean_velocity"))
			ADD_VECTOR_VALUE(mean_velocity, current_key);
		else {
			fprintf(stderr, "Unexpected key %s in %s.\n", current_key->name, filename);
			abort();
		}
	}
	CHECK_VALUE(typical_length);
	CHECK_VALUE(radius_max);
	CHECK_VALUE(total_mass);
	CHECK_VALUE(start_at);
	CHECK_VALUE(stop_at);
	realloc_gal_system(sys, stop_at+1);
	create_spherical_galaxy(sys, typical_length, radius_max, total_mass, start_at, stop_at);
	if(HAS_VALUE(center_position))
		add_position(sys, center_position, start_at, stop_at);
	if(HAS_VALUE(mean_velocity))
		add_velocity(sys, mean_velocity, start_at, stop_at);
	return 0;
}

/*
 * Parse the [uniform_gas] section.
 *
 * It has the following keys:
 * - typical_velocity
 * - radius_max
 * - total_mass
 * - start_at
 * - stop_at
 * - center_position (optional)
 * - mean_velocity (optional)
 */
int parse_uniform_gas_section(gal_system *sys, section *sec, char *filename) {
	dlist *keys = sec->keys;

	key *current_key;
	double typical_velocity;
	double radius_max;
	double total_mass;
	uint start_at;
	uint stop_at;
	double center_position[DIM];
	double mean_velocity[DIM];

	INIT_VALUE(typical_velocity);
	INIT_VALUE(radius_max);
	INIT_VALUE(total_mass);
	INIT_VALUE(start_at);
	INIT_VALUE(stop_at);
	INIT_VALUE(center_position);
	INIT_VALUE(mean_velocity);

	while((current_key = dlist_pop(keys)) != NULL) {
		if(!strcmp(current_key->name, "typical_velocity"))
			ADD_SCALAR_VALUE(typical_velocity, current_key);
		else if(!strcmp(current_key->name, "radius_max"))
			ADD_SCALAR_VALUE(radius_max, current_key);
		else if(!strcmp(current_key->name, "total_mass"))
			ADD_SCALAR_VALUE(total_mass, current_key);
		else if(!strcmp(current_key->name, "start_at"))
			ADD_SCALAR_VALUE(start_at, current_key);
		else if(!strcmp(current_key->name, "stop_at"))
			ADD_SCALAR_VALUE(stop_at, current_key);
		else if(!strcmp(current_key->name, "center_position"))
			ADD_VECTOR_VALUE(center_position, current_key);
		else if(!strcmp(current_key->name, "mean_velocity"))
			ADD_VECTOR_VALUE(mean_velocity, current_key);
		else {
			fprintf(stderr, "Unexpected key %s in %s.\n", current_key->name, filename);
			abort();
		}
	}
	CHECK_VALUE(typical_velocity);
	CHECK_VALUE(radius_max);
	CHECK_VALUE(total_mass);
	CHECK_VALUE(start_at);
	CHECK_VALUE(stop_at);
	realloc_gal_system(sys, stop_at+1);
	create_uniform_gas(sys, typical_velocity, radius_max, total_mass, start_at, stop_at);
	if(HAS_VALUE(center_position))
		add_position(sys, center_position, start_at, stop_at);
	if(HAS_VALUE(mean_velocity))
		add_velocity(sys, mean_velocity, start_at, stop_at);
	return 0;
}

/*
 * Parse the [particle] section.
 *
 * It has the following keys:
 * - position
 * - velocity
 * - mass
 * - at
 */
int parse_particle_section(gal_system *sys, section *sec, char *filename) {
	dlist *keys = sec->keys;

	key *current_key;
	double position[DIM];
	double velocity[DIM];
	double mass;
	uint at;

	INIT_VALUE(position);
	INIT_VALUE(velocity);
	INIT_VALUE(mass);
	INIT_VALUE(at);
	while((current_key = dlist_pop(keys)) != NULL) {
		if(!strcmp(current_key->name, "position"))
			ADD_VECTOR_VALUE(position, current_key);
		else if(!strcmp(current_key->name, "velocity"))
			ADD_VECTOR_VALUE(velocity, current_key);
		else if(!strcmp(current_key->name, "mass"))
			ADD_SCALAR_VALUE(mass, current_key);
		else if(!strcmp(current_key->name, "at"))
			ADD_SCALAR_VALUE(at, current_key);
		else {
			fprintf(stderr, "Unexpected key %s in %s.\n", current_key->name, filename);
			abort();
		}
	}
	CHECK_VALUE(position);
	CHECK_VALUE(velocity);
	CHECK_VALUE(mass);
	CHECK_VALUE(at);
	realloc_gal_system(sys, at+1);
	create_particle(sys, position, velocity, mass, at);
	return 0;
}

/*
 * Parse the file and return a gal_system accordingly.
 *
 * At first, we call a generic .ini parser. Then, we parse each section.
 * Each section will set up the stars from the parameter parsed.
 * There must be one [global] section.
 *
 * Examples of input file may be found in the input/ directory.
 */
gal_system *gal_system_from_file(char *filename) {
	gal_system *sys;
	FILE *fh;
	if(filename)
		fh = fopen(filename, "r");
	else
		fh = NULL;

	dlist *section_list = parse(fh);
	section *current_section;
	
	current_section = dlist_pop(section_list);
	if(current_section == NULL) {
		fprintf(stderr, "No section in %s.\n", filename);
		abort();
	}
	if(strcmp(current_section->head,"global") != 0) {
		fprintf(stderr, "File %s should start with a global section.\n", filename);
		abort();
	}

	sys = parse_global_section(current_section, filename);
	section_free(current_section);
	while((current_section = dlist_pop(section_list)) != NULL) {
		if(!strcmp(current_section->head, "hernquist_galaxy"))
			parse_hernquist_galaxy_section(sys, current_section, filename);
		else if(!strcmp(current_section->head, "uniform_gas"))
			parse_uniform_gas_section(sys, current_section, filename);
		else if(!strcmp(current_section->head, "particle"))
			parse_particle_section(sys, current_section, filename);
		else {
			fprintf(stderr, "Unexpected section %s in %s.\n", current_section->head, filename);
			abort();
		}
		section_free(current_section);
	}
	return sys;
}

