/*
 * Here we generate numbers according to certain probability distribution.
 *
 * If we want to generate numbers with the probability distribution f, we
 * have to generate a random number through H(x) where H is the inverse of F,
 * the cumulative distribution function.
 *
 * When we can't invert the cumulative distribution function, we use
 * acception-rejection method.
 */
#include "random_coord.h"
#include <math.h>

#define G 6.67300E-11
#define PI 3.14159265
#define SQ(X) ((X)*(X))

/*
 * Three functions to generate spherical coordinates according to the Hernquist model
 */

double random_hernquist_radius(double typical_length, double radius_max) {
	double uniform_rand_max;	// The superior boundary of the following random variable
	double uniform_rand_var;	// The random variable associated to the uniform law in [0,uniform_rand_max]

	uniform_rand_max = SQ(radius_max) / SQ(radius_max + typical_length);
	uniform_rand_var = random_plain(0, uniform_rand_max);

	return typical_length * sqrt(uniform_rand_var) / (1 - sqrt(uniform_rand_var));
}

double random_plain_phi() {
	return random_plain(0, 2*PI);
}

double random_plain_theta() {
	return acos(random_plain(-1, 1));
}

/*
 * Compute the probability density function of the Hernquist model
 */

double compute_hernquist_probability_density_function(double normalized_energy) {
	double q = normalized_energy;
	return ( 3*asin(q) + q*sqrt(1-SQ(q))*(1-2*SQ(q))*(8*pow(q,4)-8*SQ(q)-3) ) / pow(1-SQ(q),5/2);
}

/*
 * Compute the linear envelop density function for the following acception-rejection method
 */

double compute_linear_envelop_probability_function(double normalized_energy, double normalized_energy_max, double density_function_max) {
	return density_function_max * normalized_energy / normalized_energy_max;
}

/*
 * Generate norms of velocities, knowing the position, according to the Hernquist model
 * The acception-rejection is employed, with a linear envelop function (the Hernquist distribution is convexe)
 */

double random_hernquist_velocity_norm(double radius, double typical_length, double total_mass) {

	// First, we define the two random variable that we need for the acception-rejection method
	double random_normalized_energy;	// Associated with the linear envelop law 
	double random_test_value;		// Associated with the uniform law in [0,1]
	
	// Then, we define the range of the different variables
	double normalized_energy_max = sqrt( radius / (radius+typical_length) );
	double density_function_max = compute_hernquist_probability_density_function(normalized_energy_max);
	double test_value_max;

	// We can now compute our random variables
	int rejected = 1;
	while( rejected ) {
		random_normalized_energy = normalized_energy_max * sqrt( rand() / (double) RAND_MAX );
		random_test_value = ( rand() / (double) RAND_MAX );
		
		test_value_max = compute_hernquist_probability_density_function(random_normalized_energy) / compute_linear_envelop_probability_function(random_normalized_energy, normalized_energy_max, density_function_max);
		if( random_test_value <= test_value_max ) {
			rejected = 0;
		}
	}
	
	// Then, we retrieve the norm of the velocity v
	double vel_max = sqrt( 2*G*total_mass / (radius+typical_length) );
	return vel_max * sqrt( 1 - SQ( random_normalized_energy / normalized_energy_max ) );
}


/*
 * Compute a norm following Boltmann-like statistics.
 */
double random_boltzmann_norm(double typical_norm) {
	return typical_norm*sqrt(-log(random_plain(0, 1)));
}

/*
 * Generate exponential statistics.
 */
double random_exponential(double typical_val) {
	return -typical_val*log(random_plain(0,1));
}

/*
 * Generate a norm in order to have a plain distribution of vector in
 * spherical space.
 */
double random_plain_norm(double max_norm) {
	return max_norm*pow(random_plain(0,1),1.0/3.0);
}
