#!/usr/bin/perl
use strict;
use warnings;
use feature 'say';

my $G = 6.67398E-11;

my $output_base = './impact';
my $typical_radius = 1E18;
my $radius_max = 8E18;
my $total_mass = 1.9891E39;
my $nb_element = 2048;
my $nb_step = 3000;

my $typical_period = sqrt($typical_radius^3/($G*$total_mass));
my $typical_speed = $typical_radius/$typical_period;


my $distance = 2*$radius_max;
my $impact_parameter = 0*$typical_radius;

my $part_mass = 100*$total_mass/$nb_element;
my $part_speed = $typical_speed/2;

my $duration = 3*$distance/$part_speed;

my $stop_gal = $nb_element-1;
my $input = <<EOC;
[global]
duration=$duration
nb_step=$nb_step

[hernquist_galaxy]
typical_length=$typical_radius
radius_max=$radius_max
total_mass=$total_mass
start_at=1
stop_at=$nb_element

[particle]
position=-$distance,$impact_parameter,0
velocity=$part_speed,0,0
mass=$part_mass
at=0
EOC

chomp($input);

system("echo '$input' |./gepuku -o $output_base");

my (@x, @y, @z);
my $step = $duration/$nb_step;
foreach my $i (0 .. $nb_step-1) {
	my $fn = $output_base . sprintf("/%06d.dat", $i);
	my $line = `head -n 1 $fn`;
	chomp($line);
	for($line) {
		next if(/^\s+$/);
		($x[$i], $y[$i], $z[$i]) = /^([^\s]+)\s+([^\s]+)\s+([^\s]+)\s+$/;
	}
}
open(my $out, '>', "$output_base/speed.dat");
foreach my $i (0 .. $nb_step-2) {
	my $vx = ($x[$i+1]-$x[$i])/$step;
	my $vy = ($y[$i+1]-$y[$i])/$step;
	my $vz = ($z[$i+1]-$z[$i])/$step;

	my $dist = sqrt($x[$i]**2+$y[$i]**2+$z[$i]**2);
	my $v = sqrt($vx**2+$vy**2+$vz**2);
	say $out $dist, "\t", $v;
}

close($out);
